import { requireAuth } from "@/lib/auth/get-user"
import { getSupabaseServerClient } from "@/lib/supabase/server"
import { ReportsHeader } from "@/components/reports/reports-header"
import { OverviewStats } from "@/components/reports/overview-stats"
import { LeadsChart } from "@/components/reports/leads-chart"
import { CallsChart } from "@/components/reports/calls-chart"
import { RevenueChart } from "@/components/reports/revenue-chart"
import { TopPerformers } from "@/components/reports/top-performers"

export default async function ReportsPage() {
  const user = await requireAuth()
  const supabase = await getSupabaseServerClient()

  // Get comprehensive stats
  const [
    { count: totalLeads },
    { count: totalCalls },
    { count: totalDeals },
    { data: wonDeals },
    { data: allLeads },
    { data: allCalls },
    { data: allDealsData },
    { data: agents },
  ] = await Promise.all([
    supabase.from("leads").select("*", { count: "exact", head: true }),
    supabase.from("calls").select("*", { count: "exact", head: true }),
    supabase.from("deals").select("*", { count: "exact", head: true }),
    supabase.from("deals").select("*, agent:users!deals_agent_id_fkey(full_name)").eq("stage", "closed-won"),
    supabase.from("leads").select("created_at, status").order("created_at", { ascending: false }).limit(100),
    supabase
      .from("calls")
      .select("created_at, call_status, outcome")
      .order("created_at", { ascending: false })
      .limit(100),
    supabase.from("deals").select("amount, created_at, stage, agent_id").order("created_at", { ascending: false }),
    user.role !== "agent" ? supabase.from("users").select("id, full_name").eq("role", "agent") : { data: [] },
  ])

  // Calculate revenue
  const totalRevenue = wonDeals?.reduce((sum, deal) => sum + Number(deal.amount), 0) || 0
  const avgDealSize = wonDeals && wonDeals.length > 0 ? totalRevenue / wonDeals.length : 0

  // Calculate conversion rate
  const conversionRate = totalLeads && totalDeals ? ((wonDeals?.length || 0) / totalLeads) * 100 : 0

  // Top performers (for admins and supervisors)
  const performanceData = agents?.map((agent) => {
    const agentDeals = wonDeals?.filter((d) => d.agent_id === agent.id) || []
    const agentRevenue = agentDeals.reduce((sum, deal) => sum + Number(deal.amount), 0)
    return {
      name: agent.full_name,
      deals: agentDeals.length,
      revenue: agentRevenue,
    }
  })

  return (
    <div className="space-y-6">
      <ReportsHeader />
      <OverviewStats
        totalLeads={totalLeads || 0}
        totalCalls={totalCalls || 0}
        totalDeals={totalDeals || 0}
        totalRevenue={totalRevenue}
        avgDealSize={avgDealSize}
        conversionRate={conversionRate}
      />
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <LeadsChart leads={allLeads || []} />
        <CallsChart calls={allCalls || []} />
      </div>
      <RevenueChart deals={allDealsData || []} />
      {user.role !== "agent" && performanceData && performanceData.length > 0 && (
        <TopPerformers performers={performanceData} />
      )}
    </div>
  )
}
