import { requireAuth } from "@/lib/auth/get-user"
import { getSupabaseServerClient } from "@/lib/supabase/server"
import { PipelineBoard } from "@/components/pipeline/pipeline-board"
import { PipelineHeader } from "@/components/pipeline/pipeline-header"
import { PipelineStats } from "@/components/pipeline/pipeline-stats"

export default async function PipelinePage() {
  const user = await requireAuth()
  const supabase = await getSupabaseServerClient()

  // Get deals based on user role
  let dealsQuery = supabase
    .from("deals")
    .select(
      `
      *,
      lead:leads!deals_lead_id_fkey(first_name, last_name, phone, email),
      product:products!deals_product_id_fkey(name, price),
      agent:users!deals_agent_id_fkey(full_name)
    `,
    )
    .order("created_at", { ascending: false })

  if (user.role === "agent") {
    dealsQuery = dealsQuery.eq("agent_id", user.id)
  }

  const { data: deals } = await dealsQuery

  // Calculate stats
  const totalValue = deals?.reduce((sum, deal) => sum + Number(deal.amount), 0) || 0
  const wonDeals = deals?.filter((d) => d.stage === "closed-won") || []
  const wonValue = wonDeals.reduce((sum, deal) => sum + Number(deal.amount), 0)

  const avgDealSize = deals && deals.length > 0 ? totalValue / deals.length : 0
  const conversionRate = deals && deals.length > 0 ? (wonDeals.length / deals.length) * 100 : 0

  return (
    <div className="space-y-6">
      <PipelineHeader />
      <PipelineStats
        totalValue={totalValue}
        wonValue={wonValue}
        avgDealSize={avgDealSize}
        conversionRate={conversionRate}
        dealsCount={deals?.length || 0}
      />
      <PipelineBoard deals={deals || []} />
    </div>
  )
}
